/**
 * Arduino-Sketch zur Servopneumatischen Steuerung eines Hebels, 
 * der mit einem 60mm-ft-Pneumatik-Zylinder angestuert wird
 *
 * Servoventil:     Servobetriebenes Fischertechnik-Quetschventil
 * Servo:           Pin 9
 * 
 * 
 * Winkelmessung:   i2c Mag. Position sensor: Grove AS5600
 *                  Die Winkelmessung kann auch mit einem Quadratuir-Encoder oder einem Potentiometer 
 *                  vorgenommen werden; Das Programm ist dafür anzupassen 
 *                  (siehe dazu die verschiendenen Vorschläge für readAngle() )
 *                  
 *                  i2C-Weiche mit TCA 9548A (optional) - falls mehrere Encoder ausgelesen werden sollen
 *                  soll die Weiche verwendet werden, ist #define I2C_WEICHE zu setzen
 * 
 * 
 * Arduino-Konsole Kommandos:
 * S <SPOS>     Stelle Servo auf Position SPOS
 * TO <POS>     Fahre Arm auf Position POS
 * K?           Zeige PID-Parameter
 * KP <p>       Setze PID Proportional-Anteil auf p
 * KD <d>       Setze PID Differenzial-Anteil auf d
 * KI <i>       Setze PID Integral-Anteil auf i
 * OFF          Regelung aus
 * 
 * Auf der Arduino-Konsole werden permanent Soll-Position, Ist-Position und  Servowert ausgegeben
 * 
 * Ermittle zunächst die Servo-Endstellungen mit S
 *    s_min (Anschlag Servoventil bei Hebelarm unten) und s_max (Anschlag Servoventil bei Hebelarm oben)
 *    Der Servo betätigt den Quetsch-Schieber des Servoventils
 *    Der Schieber hat zwei Anschläge, die durch den Servo angefahren werden müssen.
 *    Bei Werten außerhalb von [s_min,s_max] übersschreitet der Servo die Anschlags-Positionen,
 *    was unbedingt vermieden werden sollte.
 *    
 * Diese Werte im Programm unten fest hinterlegen!
 * 
 * Ermittle dann Regelparameter mit den K-Befehlen (Siehe oben)
 * und hinterlege die optimalen Werte im Programm unten !
 * Nach setzen von K-Werten wird der PID-Regler sofort auf die neuen Regler-Anteile umgestellt, Man kann damit
 * sehr bequem den PID-Regler untersuchen
 * 
 * 
 * Autor: Florian Bauer - November 2025
 */

#include "AS5600.h"
#include <Servo.h>
Servo servo;

#ifdef ISR
/**
 * Interrupt-gesteuerte Auswertung von Quadratur-Encoder-Signalen für Winkelmessung
 * Pins 50 und 51 sind die Eingänge (Arduino MEGA)
 */
volatile long enc_count = 0;


void setupISR() {
  pinMode(50, INPUT);
  pinMode(51, INPUT);

  // Aktiviert Pin-Change Interrupt für Port B
  PCICR |= (1 << PCIE0);     // Enable PCINT0 interrupt (Port B)
  PCMSK0 |= (1 << PCINT3);   // Pin 50 (PB3)
  PCMSK0 |= (1 << PCINT2);   // Pin 51 (PB2)
}


volatile bool pin50Changed = false;
volatile bool pin51Changed = false;

ISR(PCINT0_vect) {
  // Statusvergleich von Port B
 static uint8_t tmp=0;
 static int8_t lookup_table[] = {0,-1,1,0,1,0,0,-1,-1,0,0,1,0,1,-1,0};
 static uint8_t enc_val = 0;
 static uint8_t lastPortB = PINB;
 uint8_t currentPortB = PINB;

 if ((currentPortB ^ lastPortB) & (1 << PB3)) {
    pin50Changed = true; 
    tmp = ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
  }
 
  if ((currentPortB ^ lastPortB) & (1 << PB2)) {
    pin51Changed = true;
  //   Serial.print("B"); 
   tmp =   ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
  }
  
  enc_val = enc_val << 2;
  enc_val = enc_val | tmp;

  enc_count = enc_count + lookup_table[enc_val & 0b1111];
  lastPortB = currentPortB;
}
#endif

  
#include <PID_v1_bc.h>

/**
 * Variablen für den PID-Regler
 */
double  sinput, soutput , ssetpoint ;

/*
 * Servo-Limits
 */
int s_max = 50;     // Maximaler Servowert  (Hebelarm oben)
int s_min = 0;      // Minimaleer Servowert (Hebelarm unten)
/*
int s_down = 10;    // ab welchem Servowert der Hebel nach unten geht
int s_up   = 30;    // ab welchem Servowwert der Hebel nach oben geht
*/

int l_slimit= s_min ; int u_slimit=s_max; 

/*
 * PID-REGLER-Parameter
 * Default PID-Werte
 */
double kp=0.25,ki=2,kd=0.004; // quetsch

PID servoPID = PID (&sinput, &soutput, &ssetpoint, kp,ki,kd, DIRECT); 

#ifdef I2C_WEICHE
/**
 * i2C-Weiche:
 */
#define TCAADDR 0x70
void tcascanner()
{
  Serial.println("\nTCAScanner ready!");
    
    for (uint8_t t=0; t<8; t++) {
      tcaselect(t);
      Serial.print("TCA Port #"); Serial.println(t);

      for (uint8_t addr = 0; addr<=127; addr++) {
        if (addr == TCAADDR) continue;

        Wire.beginTransmission(addr);
        if (!Wire.endTransmission()) {
          Serial.print("Found I2C 0x");  Serial.println(addr,HEX);
        }
      }
    }
    Serial.println("\ndone");
}

/**
 * i2C-Kanal einstellen
 */
void tcaselect(uint8_t i) {
  if (i > 7) return;
 
  Wire.beginTransmission(TCAADDR);
  Wire.write(1 << i);
  Wire.endTransmission();  
}
#endif

AS5600 encoder;

/*
 * Winkelmessung mit AS5600 Magnet-Encoder
 */
int readAngle()
{
    return 4096-encoder.getRawAngle(); 
}

/*
 * Winkelmessung mit Potentiometer
 */
 
/*
int readAngle()
{
    return analogRead(A1); // Analog-Port
}
*/

/*
 * Winkelmessung mit Quadraturgeber
 */
/*
int readAngle()
{
    return enc_count; 
}
*/

long time0;


String cmd;              // Eingabe-Variable für Kommandos


float d=0.0, d_soll=510; // Aktuelle Position, Soll-Position
float d_old=0.0;         // vorherige Position
float t,t_old=0.0;       // Zeit des loop-Aufrufs, Zeit des vorherigen loop-Aufrufs

int control_on=1;        // PID-Regelung ein


void setup() {
  Serial.begin(115200);
#ifdef I2C_WEICHE
    tcascanner();
#endif

    /**
   * Aktiviere Servo
   */
  pinMode(9,OUTPUT);
  servo.attach(9);
  servo.write(s_min);
  servoPID.SetMode(AUTOMATIC);   //set PID in Auto mode
  servoPID.SetSampleTime(1);  // refresh rate of PID controller
  servoPID.SetOutputLimits(-(u_slimit-l_slimit) / 2, (u_slimit-l_slimit) / 2); // drehventil this is the MAX PWM value to move motor, here change in value reflect change in speed of motor.

#ifdef I2C_WEICHE
    tcaselect(0);
#endif
  d_soll = readAngle();
  ssetpoint=d_soll; // Startwert = Aktueller Wert
//setupISR();
  
}


void loop() {
  // put your main code here, to run repeatedly:
  char buf[128];
  char * ptr;
  float draw=0.0;
  
#ifdef I2C_WEICHE
  tcaselect(0);  
#endif

  draw = readAngle(); 
  d=draw;
  //d= d*0.2 + 0.8* draw; // gleitender Mittelwert (falls die Winkelmessung stark streut)
  
  t=millis();

  d_old=d; t_old=t;
  
  

  while (Serial.available()) { //Check if the serial data is available.
    delay(3);                  // a small delay
    char c = Serial.read();    // storing input data
    cmd += c;                  // accumulate each of the characters in readString
  }
  if (cmd.length() >0 ) {
    cmd.toCharArray(buf, 100);
    
    if(cmd.startsWith("TO")) {
      // Sollposition setzen und Regelung einschalten
      d_soll = atof(buf+3);
      control_on=1;
    }
    if(cmd.startsWith("OFF")) {
      // Regelung ausschalten
      time0=millis();     
      control_on=0; 
    }
     if(cmd.startsWith("S")) {
      // Servo direkt stellen und Regelung ausschalten
      ptr = strtok(buf+2, ",");
      int s =atoi(ptr);
      servo.write(s);
      control_on=0;
    }
    if(cmd.startsWith("K?")) {
      // PID-Regelparameter ausgeben
      Serial.print("KP=");
      Serial.print(kp);
      Serial.print(" KI=");
      Serial.print(ki);
      Serial.print(" KD=");
      Serial.println(kd,5);

      delay(1000);
      control_on=1;
    }
    if(cmd.startsWith("KP")) {
      // PID-Proportionalanteil setzen
      kp = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    }
    if(cmd.startsWith("KI")) {
      // PID-Integralanteil setzen
      ki = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    }
     if(cmd.startsWith("KD")) {
      // PID-Differenzialanteil setzen
      kd = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    } 
    cmd="";
  }
  
  //Serial.print("SOLL:");
  Serial.print(d_soll); 
  sinput=d;
    
  ssetpoint=d_soll ;
  Serial.print(", ");
  //Serial.print("IST:"); 
  Serial.print(d); 

 
  sinput=d; //enc_count;
  ssetpoint = d_soll;
  servoPID.Compute();

  d_old=d;
      
  soutput = (u_slimit +l_slimit) / 2 +soutput; // Quetsch
 
  Serial.print(" ");
  Serial.print(soutput);
 
  if(control_on)
    servo.write(soutput);

  Serial.print(" ");
  Serial.println(control_on);
}
