/**
 * Arduino-Sketch zur Servopneumatischen Steuerung eines Hebels, 
 * der mit einem 60mm-ft-Pneumatik-Zylinder angestuert wird
 *
 * sehr schnelle und genaue Zweipuintregelung
 *  
 * Zweipunkt-Regelung der Position mit zwei 3/2-Umschalt-Magnetventilen
 * Die Ventile werden über ein Darlington-Transistor-Array ULN 2004 angesteuert
 * Drosselventile in Zu- und Abluft
 * 
 * Experimentelles Feature: Regelung der Belüftungszeit mit Direktive DELAY_CONTROL
 * Eigentlich brauechte man ein Ventil mit 3 Zustaenden (Belueften, Entlueften, Dicht) , das man ganz dicht machen kann, 
 * was bei 3/2-Wegeventilen nicht geht. Diese schalten nur zwischen Druck und Entlueftung um;
 * wir können allerdings beide Zylinderkammern mit Druck beaufschlagen, so dass auf beiden Seiten des Kolbens Druck anliegt
 * Nachteil: hoher Stromverbrauch, Ventile werden u.U. heiss
 * 
 * 
 * /!\ ACHTUNG: Da die Ventile stanedig in Betrieb sind, koennen sie u.U. sehr heiss werden. 
 *     Daher mit der geringst moeglichen Spannung (ab 6 V) betreiben und ggf. fuer Kuehlung sorgen
 * 
 * /i\ Die Qualitaet der Regelung haengt von der Dichtigkeit des Penumatikzylinders und der Haftreibung des Kolbens im Zylinder ab
 * 
 * Die Ausgaenge des Arduino werden mit 100 Ohm-Widerstaenden an die Eingaenge des Darlington-Arrays gelegt
 * Die Magnetwicklungen werden auf der einen Seite mit 6-9V und auf der anderen Seite mit den Ausgaengen des Darlington-ICs verbunden.
 * 
 * Zylinderanschluss A  (oben)   - Mittenanschluss Ventil A
 * Zylinderanschluss B  (unten)  - Mittenanschluss Ventil B
 * Wenn der Arm zittert, muss eine Abluft- und/oder eine Zuluft-Drosselung bei beiden Ventilen vorgenommen werden
 * 
 * Winkelmessung:   i2c Mag. Position sensor: Grove AS5600
 *                  Die Winkelmessung kann auch mit einem Quadratuir-Encoder oder einem Potentiometer 
 *                  vorgenommen werden; Das Programm ist dafür anzupassen 
 *                  (siehe dazu die verschiendenen Vorschläge für readAngle() )
 *                  
 *                  i2C-Weiche mit TCA 9548A (optional) - falls mehrere Encoder oder i2C-Geräte ausgelesen werden sollen
 *                  in diesem Fall ist #define I2C_WEICHE zu setzen
 * 
 * 
 * Arduino-Konsole Kommandos:
 * S <SPOS>     Stelle Servo auf Position SPOS
 * TO <POS>     Fahre Arm auf Position POS, Regelung ein
 * DN           Arm Down
 * UP           Arm Up
 * A <0|1>      Ventil A on / off
 * B <0|1>      Ventil B on / off
 * OFF          Ventile dicht und Regelung aus
 * 
 * Auf der Arduino-Konsole werden permanent Soll-Position, Ist-Position ausgegeben
 * 
 * 
 * Autor: Florian Bauer - November 2025
 */
#include "AS5600.h"

/*
 * Pins für die Ventilsteuerung
 */
#define VALVE_A 32
#define VALVE_B 33

/*
 * Optionale I2C-Weiche fuer mehrere I2C-Devices 
 */
#define I2C_WEICHE
/*
 * Experimentelles Feature, um die Belueftungszeit zu regeln
 */
#define DELAY_CONTROL
/*
 * Beim Abschalten werden beide Kammern mit DRuck beaufschlagt
 */
#define PRESSURIZE_BOTH_WHEN_IDLE

#ifdef ISR

volatile long enc_count = 0;
void setupISR() {
  

  pinMode(50, INPUT);
  pinMode(51, INPUT);

  // Aktiviert Pin-Change Interrupt für Port B
  PCICR |= (1 << PCIE0);     // Enable PCINT0 interrupt (Port B)
  PCMSK0 |= (1 << PCINT3);   // Pin 50 (PB3)
  PCMSK0 |= (1 << PCINT2);   // Pin 51 (PB2)
}
volatile bool pin50Changed = false;
volatile bool pin51Changed = false;


ISR(PCINT0_vect) {
  // Statusvergleich von Port B
 static uint8_t tmp=0;
 static int8_t lookup_table[] = {0,-1,1,0,1,0,0,-1,-1,0,0,1,0,1,-1,0};
    static uint8_t enc_val = 0;
  static uint8_t lastPortB = PINB;
  uint8_t currentPortB = PINB;
// PB2 = 4, PB3=8

  if ((currentPortB ^ lastPortB) & (1 << PB3)) {
    pin50Changed = true; 
    tmp = ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
 //   Serial.print("A");
  }
 

  if ((currentPortB ^ lastPortB) & (1 << PB2)) {
    pin51Changed = true;
  //   Serial.print("B"); 
   tmp =   ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
  }
  
        
        enc_val = enc_val << 2;
        enc_val = enc_val | tmp;

        enc_count = enc_count + lookup_table[enc_val & 0b1111];
  lastPortB = currentPortB;
}
#endif

#ifdef I2C_WEICHE
/**
 * i2C-Weiche:
 */
#define TCAADDR 0x70
void tcascanner()
{
  Serial.println("\nTCAScanner ready!");
    
    for (uint8_t t=0; t<8; t++) {
      tcaselect(t);
      Serial.print("TCA Port #"); Serial.println(t);

      for (uint8_t addr = 0; addr<=127; addr++) {
        if (addr == TCAADDR) continue;

        Wire.beginTransmission(addr);
        if (!Wire.endTransmission()) {
          Serial.print("Found I2C 0x");  Serial.println(addr,HEX);
        }
      }
    }
    Serial.println("\ndone");
}

/**
 * i2C-Kanal einstellen
 */
void tcaselect(uint8_t i) {
  if (i > 7) return;
 
  Wire.beginTransmission(TCAADDR);
  Wire.write(1 << i);
  Wire.endTransmission();  
}
#endif

AS5600 encoder;

long time0;

String cmd; // Kommando-Puffer

#define readAngle() 4096-encoder.getRawAngle(); 
 
/*
 * Winkelmessung mit AS5600 Magnet-Encoder
 */
/* int readAngle()
{
    return 4096-encoder.getRawAngle(); 
}
*/
/*
 * Winkelmessung mit Potentiometer
 */
 
/*
int readAngle()
{
    return analogRead(A1); // Analog-Port
}
*/

/*
 * Winkelmessung mit Quadraturgeber
 */
/*
int readAngle()
{
    return enc_count; 
}
*/

float d=0.0, d_soll=510;
float vd,d_old=0.0;
float t,t_old=0.0;

int u;
int control_on=1;

void setup() {
  Serial.begin(115200);
  Serial.println("Servopneumati mit Umschaltventilen");
#ifdef I2C_WEICHE
  tcascanner();
#endif

  pinMode(VALVE_B,OUTPUT);
  //digitalWrite(VALVE_B,LOW);
   pinMode(VALVE_A,OUTPUT);
  //digitalWrite(VALVE_A,LOW);
 #ifdef I2C_WEICHE
   tcaselect(0);
 #endif 
 
   d_soll =readAngle(); // Startwert = Aktueller Wert
#ifdef ISR
setupISR();
#endif

}

void loop() {
  char buf[128];
  char * ptr;
  // put your main code here, to run repeatedly:
  float draw=0.0;
#ifdef I2C_WEICHE
  tcaselect(0);  
#endif
  draw = readAngle(); //analogRead(A14);
  d= d*0.2 + 0.8* draw;
  //d=draw;
  t=millis();
  vd=(d-d_old)/(t-t_old);
  d_old=d; t_old=t;
  
  

  while (Serial.available()) { //Check if the serial data is available.
    delay(3);                  // a small delay
    char c = Serial.read();  // storing input data
    cmd += c;         // accumulate each of the characters in readString
  }
  if (cmd.length() >0 ) {
    cmd.toCharArray(buf, 100);
    
    if(cmd.startsWith("TO")) {
      d_soll = atof(buf+3);
      control_on=1;
    }
    if(cmd.startsWith("OFF")) {

      digitalWrite(VALVE_A,LOW);
      digitalWrite(VALVE_B,LOW);
      control_on=0;
      time0=millis();      
    }
     if(cmd.startsWith("ON")) {

      digitalWrite(VALVE_A,HIGH);
      digitalWrite(VALVE_B,HIGH);
      control_on=0;
      time0=millis();      
    }
    if(cmd.startsWith("A")) {
       if( atoi(buf+1) ==1)
       {
        digitalWrite(VALVE_A,HIGH);
       }
       else
       {
        digitalWrite(VALVE_A,LOW);
       }
      control_on=0;
    }
     if(cmd.startsWith("DN")) {
       
        digitalWrite(VALVE_A,HIGH);
        digitalWrite(VALVE_B,LOW);
        control_on=0;
      
    }
        if(cmd.startsWith("UP")) {
      
        digitalWrite(VALVE_A,LOW);
        digitalWrite(VALVE_B,HIGH);
      control_on=0;
      
    }
    if(cmd.startsWith("B")) {
      if( atoi(buf+1) ==1)
       {
        digitalWrite(VALVE_B,HIGH);
       }
       else
       {
        digitalWrite(VALVE_B,LOW);
       }
      control_on=0;
    }
    cmd="";
    }
 
  
  //Serial.print(" SOLL:");
  Serial.print(d_soll); 

  Serial.print(", ");
  //Serial.print("IST:"); 
  Serial.print(d); 
  if(control_on)
  {
   
    if( abs(d-d_soll) < 2)
    {
      // Sollposition ungefähr erreicht: beide Ventile auf, um Pos zu halten
      digitalWrite(VALVE_B,HIGH);
      digitalWrite(VALVE_A,HIGH);
      /*
      Serial.print(", ");
      Serial.print(0);*/
    }
    else if( d < d_soll )
    {
      // Sollposition zu klein, Arm noch zu weit unten
      digitalWrite(VALVE_B,HIGH);  // Ventilkammer für Aufwaertsbewegung belueften, Ventilkammer fuer Abwaertsbewegung entlueften
      digitalWrite(VALVE_A,LOW);  
#ifdef DELAY_CONTROL      
      u=abs(d-d_soll) /2 ;
      if(u<40) u=18;
      delay(u); // Nach dieser Zeit wird ein Ventil wieder umgeschaltet
      //Serial.print(", ");
      //Serial.print(u);
#ifdef PRESSURIZE_BOTH_WHEN_IDLE
      digitalWrite(VALVE_A,HIGH);   
#else
      digitalWrite(VALVE_B,LOW);  
#endif
#endif
    }
    else
    {
       // Sollposition zu gross, Arm noch zu weit oben
      digitalWrite(VALVE_B,LOW); 
      digitalWrite(VALVE_A,HIGH); // Ventilkammer für Abwaertsbewegung belueften, Ventilkammer fuer Aufwaertsbewegung entlueften
#ifdef DELAY_CONTROL     
      u=abs(d-d_soll) /2 ;
      if(u<40) u=18;
      delay(u);     // Nach dieser Zeit wird ein Ventil wieder umgeschaltet
      //Serial.print(", ");
      //Serial.print(u);
#ifdef PRESSURIZE_BOTH_WHEN_IDLE
      digitalWrite(VALVE_B,HIGH);
#else
      digitalWrite(VALVE_A,LOW);
#endif

#endif 
    }
 
  }
  
  d_old=d;
  Serial.println(" ");
}
