/**
 * Arduino-Sketch zur Servopneumatischen Steuerung eines Hebels, 
 * der mit einem 60mm-ft-Pneumatik-Zylinder angestuert wird
 *
 * PID-Regelung der Position mit servogesteuertem Proportionalventil
 * geeginte fuer:  Drehrorventil, Drehschieberventil
 * 
 * Die Ventile werden über einen Modellbau-Servo angesteuert
 * Drosselventile in Zu-Luft und ggf. Abluft
 * 
 * 
 * Winkelmessung:   i2c Mag. Position sensor: Grove AS5600
 *                  Die Winkelmessung kann auch mit einem Quadratuir-Encoder oder einem Potentiometer 
 *                  vorgenommen werden; Das Programm ist dafür anzupassen 
 *         
 *                  
 *                  i2C-Weiche mit TCA 9548A (optional) - falls mehrere Encoder oder i2C-Geräte ausgelesen werden sollen
 *                  in diesem Fall ist #define I2C_WEICHE zu setzen
 * 
 * 
 * Arduino-Konsole Kommandos:
 * S <SPOS>     Stelle Servo auf Position SPOS
 * TO <POS>     Fahre Arm auf Position POS, Regelung ein
 * DN           Arm Down
 * UP           Arm Up
 * K?           Zeige PID-Parameter
 * KP <p>       Setze PID Proportional-Anteil auf p
 * KD <d>       Setze PID Differenzial-Anteil auf d
 * KI <i>       Setze PID Integral-Anteil auf i

 * OFF          Ventile dicht und Regelung aus
 * 
 * Auf der Arduino-Konsole werden permanent Soll-Position, Ist-Position ausgegeben
 * 
 * Regelparameter einstellen
 * 1. Servo-Grenzwerte mit S-Befehl ermitteln: l_slimit (unterer Anschlag) und u_slimit (oberer Anschlag) 
 * 2. Servo-Werte mit S-Befehl für Beginn der Bewegung nach unten s_dn und nach oben s_up ermitteln
 * 3. Werte im Programm fest hinterlegen
 * 
 * 4. PID-Regler-Werte mit K-Befehlen ermittlen:
 * 4.1. z.B. KP=0.1, KD=KI=0
 * 4.2. Offset mit KI reduzieren
 * 4.3. Überschwinger mit KD reduzieren
 * 
 * 5. PID-Parameter kp,ki,kd im Programm setzen
 * 
 * Autor: Florian Bauer - November 2025
 */
#include "AS5600.h"
#include <AFMotor.h>
#include <Servo.h>
Servo servo;
AF_DCMotor m1(1,MOTOR12_2KHZ); // Winde für Arm A
AF_DCMotor m2(2,MOTOR12_2KHZ); // Schneckenantrieb Arm B
AF_DCMotor m3(3,MOTOR12_2KHZ); // Drehteller
 
AF_DCMotor  mot[3]   = {  m1,m2,m3};

#define I2C_WEICHE

#ifdef ISR
volatile long enc_count = 0;


#define CLK 22 //pins definitions for TM1637 and can be changed to other ports
#define DIO 24

#define VALVE_A 32
#define VALVE_B 33

void setupISR() {
  

  pinMode(50, INPUT);
  pinMode(51, INPUT);

  // Aktiviert Pin-Change Interrupt für Port B
  PCICR |= (1 << PCIE0);     // Enable PCINT0 interrupt (Port B)
  PCMSK0 |= (1 << PCINT3);   // Pin 50 (PB3)
  PCMSK0 |= (1 << PCINT2);   // Pin 51 (PB2)
}
volatile bool pin50Changed = false;
volatile bool pin51Changed = false;


ISR(PCINT0_vect) {
  // Statusvergleich von Port B
 static uint8_t tmp=0;
 static int8_t lookup_table[] = {0,-1,1,0,1,0,0,-1,-1,0,0,1,0,1,-1,0};
    static uint8_t enc_val = 0;
  static uint8_t lastPortB = PINB;
  uint8_t currentPortB = PINB;
// PB2 = 4, PB3=8

  if ((currentPortB ^ lastPortB) & (1 << PB3)) {
    pin50Changed = true; 
    tmp = ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
 //   Serial.print("A");
  }
 

  if ((currentPortB ^ lastPortB) & (1 << PB2)) {
    pin51Changed = true;
  //   Serial.print("B"); 
   tmp =   ((currentPortB & (1 << PB2) ) | (currentPortB & (1 << PB3)))>> 2;
  }
  
        
        enc_val = enc_val << 2;
        enc_val = enc_val | tmp;

        enc_count = enc_count + lookup_table[enc_val & 0b1111];
  lastPortB = currentPortB;
}
#endif
    
#include <PID_v1_bc.h>


double  sinput, soutput , ssetpoint ;
/**
 * Servo-Limits
 */
int l_slimit = 0;
int u_slimit = 60;
int s_up=5;   // Servowerte kleiner fuer Heben
int s_dn=25;  // Servowerte groesser fuer Senken

/**
 * PID-Regler-Parameter
 */
double kp=0.3, kd=0.002, ki=0.3;
PID servoPID = PID (&sinput, &soutput, &ssetpoint, kp,ki,kd, DIRECT); //0.55 20 0.0036


#ifdef I2C_WEICHE  
/**
 * i2C-Weiche:
 */
#define TCAADDR 0x70
void tcascanner()
{
  Serial.println("\nTCAScanner ready!");
    
    for (uint8_t t=0; t<8; t++) {
      tcaselect(t);
      Serial.print("TCA Port #"); Serial.println(t);

      for (uint8_t addr = 0; addr<=127; addr++) {
        if (addr == TCAADDR) continue;

        Wire.beginTransmission(addr);
        if (!Wire.endTransmission()) {
          Serial.print("Found I2C 0x");  Serial.println(addr,HEX);
        }
      }
    }
    Serial.println("\ndone");
}

/**
 * i2C-Kanal einstellen
 */
void tcaselect(uint8_t i) {
  if (i > 7) return;
 
  Wire.beginTransmission(TCAADDR);
  Wire.write(1 << i);
  Wire.endTransmission();  
}
#endif

AS5600 encoder;

long time0;



String cmd;

int dialstate=0;


float d=0.0, d_soll=510;
float vd,d_old=0.0;
float t,t_old=0.0;


int u;
int control_on=1;


void setup() {
  Serial.begin(115200);
    tcascanner();
  // put your setup code here, to run once:
  pinMode(VALVE_B,OUTPUT);
  //digitalWrite(VALVE_B,LOW);
   pinMode(VALVE_A,OUTPUT);
  //digitalWrite(VALVE_A,LOW);
    /**
   * Aktiviere Greifer-Servo
   */
  pinMode(9,OUTPUT);

  servo.attach(9);
  servoPID.SetMode(AUTOMATIC);   //set PID in Auto mode
  servoPID.SetSampleTime(1);  // refresh rate of PID controller
  servoPID.SetOutputLimits(-(u_slimit-l_slimit) / 2, (u_slimit-l_slimit) / 2); // drehventil this is the MAX PWM value to move motor, here change in value reflect change in speed of motor.
#ifdef I2C_WEICHE
  tcaselect(0);
#endif

  d_soll=ssetpoint=4096-encoder.getRawAngle(); // Startwert = Aktueller Wert
//setupISR();
  
}


void loop() {
  char buf[128];
  char * ptr;
  // put your main code here, to run repeatedly:
  float draw=0.0;
#ifdef I2C_WEICHE
  tcaselect(0);  
#endif
  draw = 4096-encoder.getRawAngle();  //analogRead(A14);
  d= d*0.2 + 0.8* draw;
 
  t=millis();
  vd=(d-d_old)/(t-t_old);
  d_old=d; t_old=t;
  
  

  while (Serial.available()) { //Check if the serial data is available.
    delay(3);                  // a small delay
    char c = Serial.read();  // storing input data
    cmd += c;         // accumulate each of the characters in readString
  }
  if (cmd.length() >0 ) {
    cmd.toCharArray(buf, 100);
    
    if(cmd.startsWith("TO")) {
      d_soll = atof(buf+3);
      control_on=1;
    }
    if(cmd.startsWith("OFF")) {
      control_on=0;
      time0=millis();      
    }
     if(cmd.startsWith("S")) {
      ptr = strtok(buf+2, ",");
      int s =atoi(ptr);
      servo.write(s);
      control_on=0;
    }
    if(cmd.startsWith("UP")) {
       servo.write(s_up);
    }
    if(cmd.startsWith("DN")) {
       servo.write(s_dn);
    }
    if(cmd.startsWith("K?")) {
      // PID-Regelparameter ausgeben
      Serial.print("KP=");
      Serial.print(kp);
      Serial.print(" KI=");
      Serial.print(ki);
      Serial.print(" KD=");
      Serial.println(kd,5);

      delay(1000);
      control_on=1;
    }
    if(cmd.startsWith("KP")) {
      // PID-Proportionalanteil setzen
      kp = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    }
    if(cmd.startsWith("KI")) {
      // PID-Integralanteil setzen
      ki = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    }
     if(cmd.startsWith("KD")) {
      // PID-Differenzialanteil setzen
      kd = atof(buf+3);
      servoPID.SetTunings(kp,ki,kd);
      control_on=1;
    } 
    cmd="";
  }

  //Serial.print(" SOLL:");
  Serial.print(d_soll); 
  sinput=d;
    
  ssetpoint=d_soll ;
  Serial.print(", ");
  //Serial.print("IST:"); 
  Serial.print(d); 
  
  sinput=d; //enc_count;
  ssetpoint = d_soll;

  servoPID.Compute();
  d_old=d;
      
  //if (soutput<0) soutput *= 0.35;
  Serial.print(" "); Serial.print(soutput); Serial.print(" ");
  if(soutput>0) soutput = s_up + soutput ; 
  if(soutput<0) soutput = s_dn + soutput; 

  Serial.print(" ");
  Serial.println(soutput);

  if(control_on)
    servo.write(soutput);
}


/**
 * PWM-Wert out für Motor i ausgeben
 */
void pwmOut(int i,int out) {            
    if (out > 0) {                  
      mot[i].run(FORWARD); mot[i].setSpeed(out); 
    }
    else {
      mot[i].run(BACKWARD); mot[i].setSpeed(abs(out)); 
    } 
}
